
from axipy.cs.CoordSystem import CoordSystem
from axipy.da import Feature, Table, Type, GeometryCollection, LineString
from axipy.interface import AxiomaInterface
from axipy.gui import selection_manager, view_manager, MapView
from axipy.app import Notifications
from typing import Iterator, List
from PySide2.QtGui import QTransform
from .helper import ensure_editable_table, quick_envelope
from axipy.concurrent import task_manager, AxipyProgressHandler, ProgressSpecification, ProgressGuiFlags

unsupported_types_string = ['Текст']
unsupported_types = [Type.Text]


def is_allowed(feature: Feature) -> bool:
    return feature.has_geometry() and feature.geometry.type not in unsupported_types \
        or issubclass(type(feature.geometry), GeometryCollection)


def send_notification_no_supported_geometry(
        title: str, iface: AxiomaInterface):
    iface.notifications.push(title=title, text=iface.tr(
        f'Не удалось обновить ни одну геометрию. Список запрещенных типов: '
        f'{", ".join(unsupported_types_string)}.'),
        type_mesage=Notifications.Critical)


def send_notification_about_geom_conversation(
        title: str, iface: AxiomaInterface):
    iface.notifications.push(
        title=title,
        text=iface.tr(
            'Часть геометрий были '
            'преобразованы в другой тип для выполнения операции.'),
        type_mesage=Notifications.Information)


def send_notification_about_finish(
        updated_cout: int, title: str, iface: AxiomaInterface):
    if updated_cout == 0:
        send_notification_no_supported_geometry(title, iface)
        return
    iface.notifications.push(
        title=title,
        text=iface.tr(f'Было преобразовано геометрий: {updated_cout}'),
        type_mesage=Notifications.Success)


# Не зависимо от КС данных пользователь хочет чтобы геометрии отражались в текущих
# координатах окна карты
def features_to_map_view_cs(
        featuers: Iterator[Feature],
        view_cs: CoordSystem,
        iface: AxiomaInterface,
        title: str) -> List[Feature]:
    res = []  # type: List[Feature]
    # TODO: use filter()
    convert_geom_counter = 0
    for feature in featuers:
        if not is_allowed(feature):
            continue
        if feature.geometry.type == Type.Arc:
            feature.geometry = feature.geometry.to_linestring()
            convert_geom_counter += 1
        if feature.geometry is None:
            continue
        # Преобразуем всю геометрию к координатам карты
        if feature.geometry.coordsystem != view_cs:
            feature.geometry = feature.geometry.reproject(view_cs)
        res.append(feature)
    if convert_geom_counter != 0:
        send_notification_about_geom_conversation(title, iface)
    return res


# TODO: Этим двум классам нужен один базовый т.к. очень много общей логики

class HorizontalMirror:
    """
    Отразить по горизонтали
    """

    def __init__(self, title: str, iface: AxiomaInterface) -> None:
        self.title = title
        self.iface = iface

    def __horizontal_mirror(
            self,
            handler: AxipyProgressHandler,
            editable_table: Table, view_cs: CoordSystem):
        features = features_to_map_view_cs(
            selection_manager.get_as_cursor(), view_cs, self.iface, self.title)
        if len(features) == 0:
            send_notification_no_supported_geometry(self.title, self.iface)
            return
        # Получаем границы геометрии в координатах карты
        full_bound = quick_envelope([f.geometry for f in features])
        dy = full_bound.height / 2 + full_bound.ymin
        transform = QTransform().fromTranslate(0, 2 * dy)
        transform = transform.scale(1, -1)
        handler.set_max_progress(len(features))
        for feature in features:
            handler.raise_if_canceled()
            geom = feature.geometry.affine_transform(transform)
            feature.geometry = geom
            handler.add_progress(1)
        handler.prepare_to_write_changes()
        editable_table.update(features)
        send_notification_about_finish(len(features), self.title, self.iface)

    def on_triggered(self):
        editable_table = ensure_editable_table()
        map_view = view_manager.active  # type: MapView
        spec = ProgressSpecification(
            description=self.title,
            flags=ProgressGuiFlags.CANCELABLE)
        task_manager.run_and_get(
            spec,
            self.__horizontal_mirror,
            editable_table,
            map_view.coordsystem)


class VerticalMirror:
    """
    Отразить по вертикали
    """

    def __init__(self, title: str, iface: AxiomaInterface) -> None:
        self.title = title
        self.iface = iface

    def __vertical_mirror(
            self,
            handler: AxipyProgressHandler,
            editable_table: Table, view_cs: CoordSystem):
        features = features_to_map_view_cs(
            selection_manager.get_as_cursor(), view_cs, self.iface, self.title)
        if len(features) == 0:
            send_notification_no_supported_geometry(self.title, self.iface)
            return
        full_bound = quick_envelope([f.geometry for f in features])

        dx = full_bound.width / 2 + full_bound.xmin
        transform = QTransform().fromTranslate(2 * dx, 0)
        transform = transform.scale(-1, 1)
        handler.set_max_progress(len(features))
        for feature in features:
            handler.raise_if_canceled()
            geom = feature.geometry
            feature.geometry = geom.affine_transform(transform)
            handler.add_progress(1)
        handler.prepare_to_write_changes()
        editable_table.update(features)
        send_notification_about_finish(len(features), self.title, self.iface)

    def on_triggered(self):
        map_view = view_manager.active  # type: MapView
        editable_table = ensure_editable_table()
        spec = ProgressSpecification(
            description=self.title,
            flags=ProgressGuiFlags.CANCELABLE)
        task_manager.run_and_get(
            spec,
            self.__vertical_mirror,
            editable_table,
            map_view.coordsystem)
