from PySide2.QtCore import QRectF, QPointF
from .Pnt import Pnt
from typing import Union


class Rect:
    """Прямоугольник, который не обладает геопривязкой. Используется для различного вида запросов."""

    def __init__(self, xmin: float, ymin: float, xmax: float, ymax: float):
        self.__rect = QRectF(xmin, ymin, xmax - xmin, ymax - ymin).normalized()

    @classmethod
    def create_empty(cls):
        obj = cls.__new__(cls)
        obj.__rect = QRectF()
        return obj

    @classmethod
    def from_qt(cls, r: QRectF):
        if not isinstance(r, QRectF):
            return None
        obj = cls.__new__(cls)
        obj.__rect = r
        return obj

    def to_qt(self):
        return self.__rect

    @classmethod
    def _rect_value_to_qt(cls, v: Union[QRectF, tuple, 'Rect']) -> QRectF:
        if isinstance(v, tuple) and len(v) == 4:
            return QRectF(QPointF(v[0], v[1]), QPointF(v[2], v[3]))
        elif isinstance(v, Rect):
            return v.to_qt()
        return v

    @classmethod
    def _rect_value_from_qt(cls, v: Union[QRectF, 'Rect']) -> 'Rect':
        if isinstance(v, QRectF):
            return Rect.from_qt(v)
        return v

    @property
    def xmin(self) -> float:
        """Минимальное значение X."""
        return self.__rect.left()

    @xmin.setter
    def xmin(self, v: float):
        return self.__rect.setLeft(v)

    @property
    def ymin(self) -> float:
        """Минимальное значение Y."""
        return self.__rect.top()

    @ymin.setter
    def ymin(self, v: float):
        return self.__rect.setTop(v)

    @property
    def xmax(self) -> float:
        """Максимальное значение X."""
        return self.__rect.right()

    @xmax.setter
    def xmax(self, v: float):
        return self.__rect.setRight(v)

    @property
    def ymax(self) -> float:
        """Максимальное значение Y."""
        return self.__rect.bottom()

    @ymax.setter
    def ymax(self, v: float):
        return self.__rect.setBottom(v)

    @property
    def center(self) -> Pnt:
        """Центр прямоугольника."""
        return Pnt.from_qt(self.__rect.center())

    @center.setter
    def center(self, p: Pnt):
        self.__rect.moveCenter(Pnt._fixPnt(p).to_qt())

    def contains(self, p: Pnt) -> bool:
        """Содержит ли в своих пределах точку.

        Args:
            p: Тестируемая точка.
        """
        return self.__rect.contains(p._point)

    def __eq__(self, other):
        import math
        if isinstance(other, Rect):
            return self.__rect == other.__rect
        return False

    def __str__(self):
        return "({} {}) ({} {})".format(self.xmin, self.ymin, self.xmax, self.ymax)

    @property
    def is_empty(self):
        """Если один или оба размера равны нулю."""
        return self.__rect.isEmpty()

    @property
    def is_valid(self):
        """Является ли прямоугольник правильным."""
        return self.__rect.isValid()

    def normalize(self):
        """Исправляет прямоугольник, если его ширина или высота отрицательны. """
        self.__rect = self.__rect.normalized()

    @property
    def width(self):
        """Ширина прямоугольника."""
        return self.__rect.width()

    @property
    def height(self):
        """Высота прямоугольника."""
        return self.__rect.height()
