from typing import Union
from PySide2.QtWidgets import QWidget
from PySide2.QtCore import Signal, QObject
from axipy.decorator import InitOnce
from .gui_class import gui_instance
from axipy.cpp_gui import ShadowAcceptableActiveToolPanelHandler, ShadowActiveToolPanelHandlerBase
from axipy.da import DefaultKeys


class ActiveToolPanelHandlerBase(QObject):

    """
    Базовый класс обработчика панели активного инструмента.
    """

    def __init__(self, shadow_handler) -> None:
        super().__init__()
        self.shadow = shadow_handler
        self._shadow_widget = None

    def __del__(self):
        self.deactivate()

    def activate(self):
        """
        Показывает пользовательский графический элемент в панели активного 
        инструмента.
        """
        self.shadow.set_widget(self._shadow_widget)
        self.shadow.activate()

    def deactivate(self):
        """
        Скрывает пользовательский графический элемент из панели активного инструмента.
        """
        self.shadow.deactivate()

    def set_panel_title(self, title: str):
        """ 
        Устанавливает заголовок панели активного инструмента. 

        Args:
            title: Новый заголовок.
        """
        self.shadow.set_panel_title(title)

    def widget(self) -> QWidget:
        """
        Возвращает пользовательский графический элемент.

        Return:
            Переданный ранее пользовательский графический элемент.
        """
        return self.shadow.widget()

    def set_observer(self, observer_id: Union[str, DefaultKeys.Key]):
        """
        Метод устанавливает наблюдателя. Если наблюдатель сигнализирует, что
        условия доступности кнопки нарушены, то панель активного инструмента
        сразу же закроется.

        Args:
            observer_id: Идентификатор наблюдателя для управления видимостью и доступностью
            
        .. seealso:: Наблюдатели за состоянием инструмента :class:`~axipy.da.observers`
        """
        is_supported = isinstance(
            observer_id,
            str) or isinstance(
            observer_id,
            DefaultKeys.Key)
        if not is_supported:
            raise TypeError(
                self.tr(f"Неподдерживаемый тип наблюдателя {type(observer_id)}"))
        if isinstance(observer_id, DefaultKeys.Key):
            observer_id = observer_id.name.decode()
        self.shadow.set_observer(observer_id)

    @property
    def activated(self) -> Signal:
        """
        ``Signal[]`` Сигнал испускается когда обработчик панели активного инструмента
        становится активным.
        """
        return self.shadow.activated

    @property
    def deactivated(self) -> Signal:
        """
        ``Signal[]`` Сигнал испускается когда перед тем как обработчик панели активного
        инструмента перестает быть активным.        
        """
        return self.shadow.deactivated


class AcceptableActiveToolHandler(ActiveToolPanelHandlerBase):
    """
    Конкретный тип обработчика панели активного инструмента, который используется 
    при наличии блока кнопок Применить/Отменить.
    """

    def __init__(
            self,
            shadow_handler: ShadowAcceptableActiveToolPanelHandler) -> None:
        super().__init__(shadow_handler)

    def set_widget(self, widget: QWidget):
        """
        Пользовательский виджет будет помещен в панель активного инструмента при
        активации обработчика. Владение виджетом передаётся обработчику.
        """
        self._shadow_widget = widget

    @property
    def accepted(self) -> Signal:
        """ 
        ``Signal[]`` Отсылается после того как пользователь нажал кнопку "Применить"
        в панели активного инструмента.
        """
        return self.shadow.accepted

    @property
    def rejected(self) -> Signal:
        """
        ``Signal[]`` Отсылается после того как пользователь нажал кнопку "Отмена"
        в панели активного инструмента.
        """
        return self.shadow.rejected


class ActiveToolPanel(QObject):
    """
    Сервис предоставляющий доступ к панели активного инструмента.

    .. literalinclude:: /../../tests/doc_examples/test_example_active_tool_panel.py
        :caption: Пример использования.
        :pyobject: test_make_handler
        :lines: 2-
        :dedent: 4

    Чтобы отобразить переданный ранее графический элемент нужно вызвать
    :meth:`~axipy.gui.ActiveToolPanelHandlerBase.activate`. Например при 
    нажатии на пользовательскую кнопку. 
    
    Панель активного инструмента по умолчанию содержит кнопки "Применить" 
    и "Отмена". По нажатию кнопки "Отмена" посылается сигнал 
    :meth:`~axipy.gui.AcceptableActiveToolHandler.rejected` и очищается содержимое
    панели активного инструмента. По нажатию "Применить" отсылается сигнал 
    :meth:`~axipy.gui.AcceptableActiveToolHandler.accepted`.
    """

    def __init__(self) -> None:
        super().__init__(None)

    def make_handler(self,
                     title: str,
                     observer_id: Union[str,
                                        DefaultKeys.Key],
                     widget: QWidget = None) -> AcceptableActiveToolHandler:
        """
        Создает экземпляр обработчика через который можно взаимодействовать с
        панелью активного инструмента.

        Args:
            title: Заголовок панели активного инструмента. Обычно это название инструмента.
            observer_id: Идентификатор наблюдателя для управления видимостью и доступностью.
            widget: Пользовательский виджет который будет отображаться в панели активного инструмента.
        """
        panel_handler = self.__wrap(self.__service().make_handler())
        panel_handler.set_observer(observer_id)
        panel_handler.set_panel_title(title)
        panel_handler.set_widget(widget)
        return panel_handler

    @InitOnce
    def __service(self):
        from axipy.cpp_gui import ShadowActiveToolPanelMediator
        shadow_panel = ShadowActiveToolPanelMediator(gui_instance.shadow)
        return shadow_panel

    def __wrap(self, shadow_handler: ShadowActiveToolPanelHandlerBase):
        if isinstance(shadow_handler, ShadowAcceptableActiveToolPanelHandler):
            return AcceptableActiveToolHandler(shadow_handler)
        else:
            raise NotImplementedError()
