
import sys
import traceback
from axipy.app import Notifications
from axipy.concurrent.TaskUtils import ProgressGuiFlags
from axipy.da import provider_manager, data_manager
from PySide2.QtWidgets import QApplication
from axipy.da.attribute_schema import Attribute

from axipy.interface import AxiomaInterface
from axipy.sql import get_database, geometry_uid, style_uid
from axipy.da import Table, Geometry, Feature, Style, DefaultKeys, GeometryCollection
from axipy.cs import CoordSystem
from .helper import ensure_editable_table, selection_as_tmp_table, DataManagerGuard, \
    text_to_float, ensure_sql_init_for_bg_thread
from PySide2.QtSql import QSqlQuery
from PySide2.QtUiTools import QUiLoader
from PySide2.QtWidgets import QWidget, QGridLayout
from PySide2.QtGui import QDoubleValidator
from axipy.concurrent import task_manager, AxipyProgressHandler, ProgressSpecification
from axipy.gui import view_manager, selection_manager
from typing import List


MIN_POINTS_COUNT = 2

class VoronovSettings:
    """ Настраиваемые параметры, которые можно передать при
    вызове функции из SQL  """

    def __init__(self) -> None:
        # Значения по умолчанию взяты из документации к SpatiaLite
        self.edges_only = False  # type: bool
        self.frame_extra_size = 5  # type: int
        self.tolerance = 0  # type: float


class PolygonsVoronogoSettingsWidget(QWidget):

    def __init__(self, iface: AxiomaInterface, ui_file: str) -> None:
        super().__init__()
        self.__iface = iface
        self.__ui = QUiLoader().load(ui_file)
        self.__ui.setParent(self)
        grid_layout = QGridLayout()
        grid_layout.addWidget(self.__ui)
        self.setLayout(grid_layout)
        self.init_tolerance_le()

    def init_tolerance_le(self):
        validator = QDoubleValidator(
            bottom=sys.float_info.min,
            top=sys.float_info.max,
            decimals=15)
        validator.setNotation(QDoubleValidator.StandardNotation)
        self.__ui.tolerance_le.setValidator(validator)
        self.__ui.tolerance_le.setToolTip(self.__iface.tr(
            "Если координаты точек отличаются друг от друга меньше чем на введенное "
            "значение, то они не будут учитываться при выполнении операции."))

    def update_coordinate_unit(self, cs: CoordSystem):
        self.__ui.unit_tolerance_label.setText(cs.unit.localized_name)

    def get_settings(self) -> VoronovSettings:
        result = VoronovSettings()
        result.edges_only = self.__ui.edges_only.isChecked()
        result.frame_extra_size = self.__ui.frame_extra_size_sb.value()
        if self.__ui.tolerance_gb.isChecked():
            result.tolerance = text_to_float(self.__ui.tolerance_le.text(),
                                             self.__ui.tolerance_le.locale())
        return result


class PolygonsVoronogo:
    """ Построение полигонов Вороного по выбранным точкам """

    def __init__(
            self,
            title: str,
            iface: AxiomaInterface,
            observer_id: DefaultKeys.Key) -> None:
        self.title = title
        self.iface = iface
        ui_file = self.iface.local_file('ui', 'VoronovPolygon.ui')
        panel_service = self.iface.active_tool_panel()
        widget = PolygonsVoronogoSettingsWidget(self.iface, ui_file)
        self.__tool_panel = panel_service.make_handler(
            title=self.title,
            observer_id=observer_id,
            widget=widget)
        self.__tool_panel.accepted.connect(self.__run_in_background_thread)

        def update_coordinate_unit():
            widget.update_coordinate_unit(selection_manager.table.coordsystem)
        self.__tool_panel.activated.connect(update_coordinate_unit)
        self.__tmp_tables = []  # type: List[Table]

    def __close_tmp_tables(self):
        for table in self.__tmp_tables:
            if table is None:
                continue
            table.restore()
            table.close()

    def on_triggered(self) -> None:
        self.__tool_panel.activate()
        ensure_sql_init_for_bg_thread()

    def __run_in_background_thread(self):
        settings = self.__tool_panel.widget().get_settings()
        selection_table = selection_as_tmp_table()
        try:
            ph = AxipyProgressHandler()
            self.__make_voronov_polygons(ph,
                settings,
                selection_table,
                ensure_editable_table())
        except BaseException:
            traceback.print_exc()
        finally:
            self.__close_tmp_tables()

    def __make_voronov_polygons(
            self,
            handler: AxipyProgressHandler,
            settings: VoronovSettings,
            selection_table_guard: DataManagerGuard,
            # selection_table: Table,
            editable_table: Table):
        view_cs = view_manager.active.coordsystem  # type: CoordSystem
        selection_table = self.__reproject_selection_table(selection_table_guard.data, view_cs)
        query = QSqlQuery(get_database())
        cs_view_str = f"proj:{view_cs.proj}"
        if len(cs_view_str) == 0:
            cs_view_str = f"prj:{view_cs.prj}"
        if len(cs_view_str) == 0:
            raise RuntimeError(
                self.iface.tr("Не удаётся получить координатную систему карты"))
        # TransformToAxiGeo преобразует геометрию к проекции карты, это даёт
        # возможность построить Полигоны Вороного в проекции карты.
        query_text = f"SELECT AsBinary(VoronojDiagram(Collect(FromAxiGeo({geometry_uid})), \
            {False}, {settings.frame_extra_size}, {settings.tolerance})) \
            FROM {selection_table.name} WHERE Str$({geometry_uid}) == 'Point'"
        res = query.exec_(query_text)
        if not res or not query.next():
            print(
                self.iface.tr(f'Произошла ошибка при выполнении SQL-запроса {query_text}'))
            return
        # 0 т.к. мы запрашиваем только одну колонку, индексы начинаются с 0
        wkb = query.value(0)
        if len(wkb) == 0:
            self.__nofify_about_suitable_types()
            return
        collection = Geometry.from_wkb(
            wkb, view_cs)  # type: GeometryCollection
        handler.prepare_to_write_changes()
        handler.set_description(
            self.iface.tr("Копируем аттрибутивную информацию"))
        self.__copy_attributes_with_sql(
            handler,
            collection,
            selection_table,
            editable_table,
            settings)

    def column_names(self, attr_table: Table, geom_table: Table) -> List[str]:
        res_attrs = []
        for attr in attr_table.schema:
            res_attrs.append(f"{attr_table.name}.{attr.name}")
        res_attrs.append(f"{geom_table.name}.{geometry_uid}")
        res_attrs.append(f"{geom_table.name}.{style_uid}")
        return ', '.join(res_attrs)

    def __insert_to_table(
            self,
            handler: AxipyProgressHandler,
            editable_table: Table,
            polygons_with_attr_table: Table,
            use_linestring: bool):
        table_size = polygons_with_attr_table.count()
        if use_linestring:
            handler.set_description(
                self.iface.tr("Преобразуем полигоны в полилинии"))
            handler.set_max_progress(table_size)
            handler.set_progress(0)

            def to_linestring(f: Feature) -> Feature:
                handler.raise_if_canceled()
                f.geometry = f.geometry.to_linestring()
                f.style = Style.for_geometry(f.geometry)
                handler.add_progress(1)
                QApplication.instance().processEvents()
                return f
            features = [to_linestring(f) for f in polygons_with_attr_table.items()]
            handler.prepare_to_write_changes()
            editable_table.insert(features)
        else:
            handler.prepare_to_write_changes()
            editable_table.insert(
                [f for f in polygons_with_attr_table.items()])
        self.__notify_about_finish(table_size)

    def __notify_about_finish(self, count_features: int):
        if count_features == 0:
            self.iface.notifications.push(
                self.title,
                self.iface.tr("Не удалось добавить ни один полигон Вороного"),
                Notifications.Critical)
        else:
            self.iface.notifications.push(self.title, self.iface.tr(
                f"Было добавленно записей: {count_features}"), Notifications.Success)

    def __reproject_selection_table(self, table: Table, view_cs: CoordSystem) -> Table:
        if table.coordsystem == view_cs:
            return table
        schema = table.schema
        schema.coordsystem = view_cs
        definition = {'src': '', 'schema': schema}
        table_in_view_cs = provider_manager.create_open(definition)  # type: Table
        QApplication.instance().processEvents()
        self.__tmp_tables.append(table_in_view_cs)
        res = [] # type: List[Feature]
        for f in table.items():
            f.geometry = f.geometry.reproject(view_cs)
            res.append(f)
        table_in_view_cs.insert(res)
        return table_in_view_cs


    def __copy_attributes_with_sql(self, handler: AxipyProgressHandler,
                                   collection: GeometryCollection,
                                   selection_table: Table,
                                   editable_table: Table,
                                   settings: VoronovSettings):
        polygon_features = []  # type: List[Feature]
        for geom in collection:
            polygon_features.append(
                Feature(
                    geometry=geom,
                    style=Style.for_geometry(geom)))
        definition = {'src': '', 'schema': selection_table.schema}
        polygons_table = provider_manager.create_open(definition)  # type: Table
        self.__tmp_tables.append(polygons_table)
        # нужно обработать очередь сообщений иначе таблица будет недоступна для SQL
        QApplication.instance().processEvents()
        polygons_table.insert(polygon_features)
        columns = self.column_names(selection_table, polygons_table)
        query_text = f"SELECT {columns} FROM {polygons_table.name},{selection_table.name}" \
            f" WHERE {selection_table.name}.{geometry_uid}" \
            f" within {polygons_table.name}.{geometry_uid}"
        polygons_with_attr_table = data_manager.query(
            query_text)  # type: Table
        if polygons_with_attr_table is None:
            return
        data_manager.add(polygons_with_attr_table)
        QApplication.instance().processEvents()
        # self.__tmp_tables.append(polygons_with_attr_table)
        self.__insert_to_table(
            handler,
            editable_table,
            polygons_with_attr_table,
            settings.edges_only)

    def __nofify_about_suitable_types(self):
        self.iface.notifications.push(
            self.title, self.iface.tr(
                f"Не удалось выполнить построение Полигонов Вороного. "
                f"Поддерживаемые типы геометрий: Точки. "
                f"Минимальное колличество: {MIN_POINTS_COUNT}"), Notifications.Critical)
